function cor2bshort(subject, volid, bpres, ipres)
%
% cor2bshort(subjname, volid, bpres, ipres)
%
% Converts a corronal volume found in $SUBJECTS_DIR/subject/mri/volid
% into bshort format with bpres between-plane resolution and ipres
% in-plane resolution. When a resolution more than 1 mm, all the COR
% slices within that resolution are averaged together.
%
% The bshorts are put in % $SUBJECTS_DIR/subject/mri/volid/bshort%d%d 
% where the first %d is the bpres and the second is ipres.  Also creates 
% register.dat and analyze.dat.
%
% The bshorts can be viewed against the originals with
%   tkregister local $SUBJECTS_DIR/subject/mri/volid
%


%
% cor2bshort.m
%
% Original Author: Doug Greve
% CVS Revision Info:
%    $Author: nicks $
%    $Date: 2007/01/10 22:02:29 $
%    $Revision: 1.2 $
%
% Copyright (C) 2002-2007,
% The General Hospital Corporation (Boston, MA). 
% All rights reserved.
%
% Distribution, usage and copying of this software is covered under the
% terms found in the License Agreement file named 'COPYING' found in the
% FreeSurfer source code root directory, and duplicated here:
% https://surfer.nmr.mgh.harvard.edu/fswiki/FreeSurferOpenSourceLicense
%
% General inquiries: freesurfer@nmr.mgh.harvard.edu
% Bug reports: analysis-bugs@nmr.mgh.harvard.edu
%

if(nargin ~= 4)
  msg = 'USAGE: cor2bshort(subject, volid, bpres, ipres)';
  error(msg);
end

% between-plane resolution
resolution = bpres;
% in-plane resolution
ipresolution = ipres;

subjsdir = getenv('SUBJECTS_DIR');
indir = sprintf('%s/%s/mri/%s',subjsdir,subject,volid);
cd(indir);
outdir = sprintf('bshort%d%d',resolution,ipresolution);

[status msg] = mkdir(outdir);
if(status == 0)  error(msg); end

outstem = sprintf('%s/f',outdir);

noutslices = 256/resolution;
noutrows = 256/ipresolution;
noutcols = 256/ipresolution;

nn = 1:resolution:256;
ipnn = 1:ipresolution:256;

ncorslice = 1;
for noutslice = 1:noutslices
  fprintf('%3d ',noutslice);
  if(rem(noutslice,16) == 0) fprintf('\n'); end

  % Average slices (closer than resolution) together %
  tmpslice = 0;
  for n = 1:resolution
    corfile = sprintf('%s/COR-%03d',indir,ncorslice);
    corslice = fast_ldcorslice(corfile);
    tmpslice = tmpslice + double(corslice);
    ncorslice = ncorslice + 1;
  end
  tmpslice = tmpslice/resolution;

  % Within a slice, average the rows (closer than res) together
  tmpslice2 = 0;
  for n = 1:ipresolution
    tmpslice2 = tmpslice2 + tmpslice(ipnn+n-1,:);
  end
  tmpslice2 = tmpslice2/ipresolution;

  % Within a slice, average the columns (closer than res) together
  outslice = 0;
  for n = 1:ipresolution
    outslice = outslice + tmpslice2(:,ipnn+n-1);
  end
  outslice = outslice/ipresolution;

  outfile = sprintf('%s_%03d.bshort',outstem,noutslice-1);
  fmri_svbfile(outslice,outfile);

end

% write out the register.dat
fname = sprintf('%s/register.dat',outdir);
fid = fopen(fname,'w');
fprintf(fid,'%s\n',subject);
fprintf(fid,'%f\n',ipresolution);
fprintf(fid,'%f\n',resolution);
fprintf(fid,'1.0\n');
fprintf(fid,'%f %f %f %f \n',eye(4));
fclose(fid);

% write out the analyse.dat
fname = sprintf('%s/analyse.dat',outdir);
fid = fopen(fname,'w');
fprintf(fid,'.\n');
fprintf(fid,'f_%%03d.bshort\n');
fprintf(fid,'%d 1\n',noutslices);
fprintf(fid,'%3d %3d\n',noutrows,noutcols);
fclose(fid);

% write out a readme file %
fname = sprintf('%s/readme',outdir);
fid = fopen(fname,'w');
fprintf(fid,'README file for bshort%d%d\n',resolution,ipresolution);
fprintf(fid,'\n');
fprintf(fid,'This directory holds the same data as the CORs in the directory above \n');
fprintf(fid,'but in bshort format. However, while the CORs are 1 mm resolution, \n');
fprintf(fid,'these slices have a between-slice resolution of %d mm and an in-plane\n',...
            resolution);
fprintf(fid,'resolution of %d mm (that is where the %d%d comes from).  When the\n',...
            ipresolution,resolution,ipresolution);
fprintf(fid,'bshort resolution is less than 1 mm, the COR slices/rows/columns\n');
fprintf(fid,'within that resolution are averaged together.\n');
fprintf(fid,'\n');
fprintf(fid,'There is also a register.dat and an analyse.dat so that the volume can\n');
fprintf(fid,'be viewed in tkmedit and tkregister. To check the registration, cd to\n');
fprintf(fid,'bshort%d%d and type:\n',resolution,ipresolution);
fprintf(fid,'\n');
fprintf(fid,'tkregister local CORPath\n');
fprintf(fid,'\n');
fprintf(fid,'Generated by cor2bshort.m\n');
fprintf(fid,'\n');
fprintf(fid,'Douglas Greve\n');
fclose(fid);
