<!-- Copyright 2011 The MathWorks, Inc.

     This file converts a project in the original MATLAB Coder format to the format used by the plugin version shipping with R2012a.

     The 12a version of the plugin makes use of the new profile support in the infrastructure to largely avoid having duplicate
     params named param.grt.*, param.ert.*, etc. However, there are 3 params that still use the old mex and grt prefixes,
     because their definitions are actually different and for good reasons.

     The other major difference is that the hardware implementation params have been redefined as normal params rather than a block
     of custom XML.

     So most of the transformation logic involves chopping off the old "fake profile" prefixes and reorganizing the hardware params.
-->

<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

    <xsl:output method="xml" indent="yes" />

    <xsl:template match="configuration">
        <configuration target="target.matlab.coder">
            <profile key="profile.mex">
                <xsl:call-template name="mex" />
                <xsl:for-each select="unset">
                    <unset>
                        <xsl:call-template name="mex" />
                    </unset>
                </xsl:for-each>
            </profile>

            <profile key="profile.c">
                <xsl:call-template name="c" />
                <xsl:for-each select="unset">
                    <unset>
                        <xsl:call-template name="c" />
                    </unset>
                </xsl:for-each>
            </profile>

            <!-- Copy all non-param stuff -->
            <xsl:for-each select="*">
                <xsl:if test="name() != 'unset' and name() != 'param.hardware.implementation' and not(starts-with(name(), 'param.ert.')) and not(starts-with(name(), 'param.grt.')) and not(starts-with(name(), 'param.mex.'))">
                    <xsl:copy-of select="." />
                </xsl:if>

                <xsl:if test="contains(name(), 'outputfile') or contains(name(), 'GenCodeOnly') or contains(name(), 'mainhtml')">
                    <xsl:copy-of select="." />
                </xsl:if>
            </xsl:for-each>

            <!-- The unset param section now leaves out params that are in profiles, so we need to filter them out. -->
            <unset>
                <xsl:for-each select="unset/*">
                    <xsl:if test="name() != 'param.hardware.implementation' and not(starts-with(name(), 'param.ert.')) and not(starts-with(name(), 'param.grt.')) and not(starts-with(name(), 'param.mex.'))">
                        <xsl:copy-of select="." />
                    </xsl:if>
                </xsl:for-each>

                <xsl:for-each select="unset/*[contains(name(), '.mainhtml') or contains(name(), '.outputfile') or contains(name(), '.GenCodeOnly')]">
                    <xsl:copy-of select="." />
                </xsl:for-each>
            </unset>
        </configuration>
    </xsl:template>

    <xsl:template match="deployment-project">
        <deployment-project>
            <xsl:apply-templates select="*" />
        </deployment-project>
    </xsl:template>

    <!-- ============== Helper templates ============== -->

    <!-- Convert option.general.Foo to option.Foo -->
    <xsl:template name="stripOptionGeneral">
        <xsl:value-of select="concat('option.', substring-after(text(), 'option.general.'))" />
    </xsl:template>

    <!-- Strip an old profile prefix from a param -->
    <xsl:template name="strip">
        <xsl:param name="prefix" />
        <xsl:variable name="param-name" select="concat('param.', substring-after(name(), $prefix))" />

        <xsl:choose>
            <!-- TargetFunctionLibrary becomes CodeReplacementLibrary in R2012a -->
            <xsl:when test="contains(name(), 'TargetFunctionLibrary')">
                    <param.CodeReplacementLibrary><xsl:value-of select="text()" /></param.CodeReplacementLibrary>
            </xsl:when>

            <xsl:otherwise>
                <xsl:element name="{$param-name}">
                    <xsl:choose>
                        <!-- Some enum values go from option.general.foo to option.foo -->
                        <xsl:when test="starts-with(text(), 'option.') and (contains(name(), '.TargetLang') or contains(name(), '.FilePartitionMethod') or contains(name(), '.GlobalDataSyncMethod'))">
                            <xsl:call-template name="stripOptionGeneral" />
                        </xsl:when>

                        <xsl:when test="contains(name(), '.DynamicMemoryAllocation') and not(contains(name(), '.DynamicMemoryAllocationThreshold'))">
                            <xsl:choose>
                                <!-- These two options are used by tests, because they are the config object values -->
                                <xsl:when test=". = 'option.general.DynamicMemoryAllocation.Off'">option.DynamicMemoryAllocation.Disabled</xsl:when>
                                <xsl:when test=". = 'option.general.DynamicMemoryAllocation.AllVariableSizeArrays'">option.DynamicMemoryAllocation.Enabled</xsl:when>

                                <xsl:otherwise><xsl:call-template name="stripOptionGeneral" /></xsl:otherwise>
                            </xsl:choose>
                        </xsl:when>

                        <xsl:otherwise>
                            <xsl:copy-of select="node()" />
                        </xsl:otherwise>
                    </xsl:choose>
                </xsl:element>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:template>

    <!-- All mex params -->
    <xsl:template name="mex">
        <xsl:for-each select="*">
            <xsl:choose>
                <xsl:when test="starts-with(name(), 'param.mex.general.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.mex.general.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.mex.symbols.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.mex.symbols.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.mex.customcode.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.mex.customcode.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.mex.comments.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.mex.comments.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.mex.paths.')">
                    <xsl:call-template name="paths">
                        <xsl:with-param name="prefix" select="'param.mex.'" />
                    </xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.mex.report.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.mex.report.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.mex.') and not(contains(name(), 'outputfile')) and not(contains(name(), 'GenCodeOnly')) and not(contains(name(), 'mainhtml'))">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.mex.'" /></xsl:call-template>
                </xsl:when>
            </xsl:choose>
        </xsl:for-each>
    </xsl:template>

    <!-- All C params -->
    <xsl:template name="c">
        <xsl:for-each select="*">
            <xsl:choose>
                <!-- SupportNonFinite is the only param having both ERT and GRT variants that may conflict.
                     The default value is true; we want to make it false, and suppress it from the unset section,
                     if either has been set to false by the user. -->
                <xsl:when test="name() = 'param.ert.SupportNonFinite' or name() = 'param.grt.SupportNonFinite'">
                    <xsl:variable name="changed" select="/deployment-project/configuration/param.ert.SupportNonFinite = 'false' or /deployment-project/configuration/param.grt.SupportNonFinite = 'false'" />
                    <xsl:variable name="inUnset" select="local-name(parent::*) = 'unset'" />

                    <xsl:if test="$inUnset = false or $changed = false">
                        <param.SupportNonFinite>
                            <xsl:if test="$inUnset = false">
                                <xsl:choose>
                                    <xsl:when test="$changed">false</xsl:when>
                                    <xsl:otherwise>true</xsl:otherwise>
                                </xsl:choose>
                            </xsl:if>
                        </param.SupportNonFinite>
                    </xsl:if>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.grt.general.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.grt.general.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.grt.customcode.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.grt.customcode.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.grt.comments.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.grt.comments.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.grt.paths.')">
                    <xsl:call-template name="paths">
                        <xsl:with-param name="prefix" select="'param.grt.'" />
                    </xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.grt.report.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.grt.report.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.grt.') and not(contains(name(), 'outputfile')) and not(contains(name(), 'GenCodeOnly')) and not(contains(name(), 'mainhtml'))">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.grt.'" /></xsl:call-template>
                </xsl:when>

                <xsl:when test="starts-with(name(), 'param.ert.')">
                    <xsl:call-template name="strip"><xsl:with-param name="prefix" select="'param.ert.'" /></xsl:call-template>
                </xsl:when>

                <!-- The hardware stuff used to be custom XML; now it is normal params -->
                <xsl:when test="name() = 'param.hardware.implementation'">
                    <param.SameHardware><xsl:value-of select="HardwareImplementation/ProdEqTarget" /></param.SameHardware>

                    <xsl:call-template name="hardware">
                        <xsl:with-param name="oldPrefix" select="'Target'" />
                        <xsl:with-param name="newSuffix" select="'Target'" />
                    </xsl:call-template>

                    <xsl:call-template name="hardware">
                        <xsl:with-param name="oldPrefix" select="'Prod'" />
                        <xsl:with-param name="newSuffix" select="'Production'" />
                    </xsl:call-template>
                </xsl:when>
            </xsl:choose>
        </xsl:for-each>
    </xsl:template>

    <!-- Convert paths to use the uniform naming convention -->
    <xsl:template name="paths">
        <xsl:param name="prefix" />

        <xsl:choose>
            <xsl:when test="name() = concat($prefix, 'paths.working')">
                <param.WorkingFolder>
                    <xsl:choose>
                        <xsl:when test=". = 'option.paths.working.project'">option.WorkingFolder.Project</xsl:when>
                        <xsl:when test=". = 'option.paths.working.current'">option.WorkingFolder.Current</xsl:when>
                        <xsl:when test=". = 'option.paths.working.specified'">option.WorkingFolder.Specified</xsl:when>
                    </xsl:choose>
                </param.WorkingFolder>
            </xsl:when>

            <xsl:when test="name() = concat($prefix, 'paths.working.specified')">
                <param.SpecifiedWorkingFolder><xsl:value-of select="." /></param.SpecifiedWorkingFolder>
            </xsl:when>

            <xsl:when test="name() = concat($prefix, 'paths.build')">
                <param.BuildFolder>
                    <xsl:choose>
                        <xsl:when test=". = 'option.paths.build.project'">option.BuildFolder.Project</xsl:when>
                        <xsl:when test=". = 'option.paths.build.current'">option.BuildFolder.Current</xsl:when>
                        <xsl:when test=". = 'option.paths.build.specified'">option.BuildFolder.Specified</xsl:when>
                    </xsl:choose>
                </param.BuildFolder>
            </xsl:when>

            <xsl:when test="name() = concat($prefix, 'paths.build.specified')">
                <param.SpecifiedBuildFolder><xsl:value-of select="." /></param.SpecifiedBuildFolder>
            </xsl:when>

            <xsl:when test="name() = concat($prefix, 'paths.search')">
                <param.SearchPaths><xsl:copy-of select="node()" /></param.SearchPaths>
            </xsl:when>
        </xsl:choose>
    </xsl:template>

    <!-- Translate one generic hardware-related field -->
    <xsl:template name="hardwareField">
        <xsl:param name="oldName" />
        <xsl:param name="newName" />
        <xsl:variable name="value" select="HardwareImplementation/*[name() = $oldName]" />

        <xsl:if test="string-length($value) != 0">
            <xsl:element name="{$newName}"><xsl:value-of select="$value" /></xsl:element>
        </xsl:if>
    </xsl:template>

    <!-- Translate one hardware size-related field -->
    <xsl:template name="hardwareSizeField">
        <xsl:param name="oldPrefix" />
        <xsl:param name="newSuffix" />
        <xsl:param name="typeName" />

        <xsl:call-template name="hardwareField">
            <xsl:with-param name="oldName" select="concat($oldPrefix, 'BitPer', $typeName)" />
            <xsl:with-param name="newName" select="concat('param.HardwareSize', $typeName, '.', $newSuffix)" />
        </xsl:call-template>
    </xsl:template>

    <!-- Translate all the hardware fields for target or production -->
    <xsl:template name="hardware">
        <xsl:param name="oldPrefix" />
        <xsl:param name="newSuffix" />

        <xsl:variable name="vendorString">
            <xsl:choose>
                <xsl:when test="HardwareImplementation/*[name() = concat($oldPrefix, 'HWDeviceType')]">
                    <xsl:value-of select="substring-before(HardwareImplementation/*[name() = concat($oldPrefix, 'HWDeviceType')], '-&gt;')" />
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="HardwareImplementation/*[name() = concat($oldPrefix, 'VendorName')]" />
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="targetString">
            <xsl:choose>
                <xsl:when test="HardwareImplementation/*[name() = concat($oldPrefix, 'HWDeviceType')]">
                    <xsl:value-of select="substring-after(HardwareImplementation/*[name() = concat($oldPrefix, 'HWDeviceType')], '-&gt;')" />
                </xsl:when>
                <xsl:otherwise>
                    <xsl:value-of select="HardwareImplementation/*[name() = concat($oldPrefix, 'TypeName')]" />
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>

        <xsl:choose>
            <!-- 12a stops allowing ASIC for target hardware; it should not have been allowed before -->
            <xsl:when test="$oldPrefix = 'Target' and $targetString = 'ASIC/FPGA'">
                <param.HardwareVendor.Target>Generic</param.HardwareVendor.Target>
                <param.HardwareType.Target>MATLAB Host Computer</param.HardwareType.Target>
            </xsl:when>

		    <!-- NEC has been renamed to Renesas -->
		    <xsl:when test="$oldPrefix = 'Target' and $vendorString = 'NEC'">
		    	<param.HardwareVendor.Target>Renesas</param.HardwareVendor.Target>
		    	<param.HardwareType.Target><xsl:value-of select="$targetString" /></param.HardwareType.Target>
		    </xsl:when>
		    
		    <xsl:when test="$oldPrefix = 'Prod' and $vendorString = 'NEC'">
		    	<param.HardwareVendor.Production>Renesas</param.HardwareVendor.Production>
		    	<param.HardwareType.Production><xsl:value-of select="$targetString" /></param.HardwareType.Production>
		    </xsl:when>
		    
            <!-- If they don't have ASIC for target hardware, just copy the selection -->
            <xsl:otherwise>
                <xsl:element name="{concat('param.HardwareVendor.', $newSuffix)}"><xsl:value-of select="$vendorString" /></xsl:element>
                <xsl:element name="{concat('param.HardwareType.', $newSuffix)}"><xsl:value-of select="$targetString" /></xsl:element>
            </xsl:otherwise>
        </xsl:choose>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Char'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Short'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Int'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Long'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Float'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Double'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareSizeField">
            <xsl:with-param name="typeName" select="'Pointer'" />
            <xsl:with-param name="oldPrefix" select="$oldPrefix" />
            <xsl:with-param name="newSuffix" select="$newSuffix" />
        </xsl:call-template>

        <xsl:call-template name="hardwareField">
            <xsl:with-param name="oldName" select="concat($oldPrefix, 'WordSize')" />
            <xsl:with-param name="newName" select="concat('param.HardwareSizeWord.', $newSuffix)" />
        </xsl:call-template>

        <xsl:element name="{concat('param.HardwareEndianness.', $newSuffix)}">
            <xsl:variable name="endianness" select="HardwareImplementation/*[name() = concat($oldPrefix, 'Endianess')]" />

            <xsl:choose>
                <xsl:when test="$endianness = '0'">option.HardwareEndianness.Unspecified</xsl:when>
                <xsl:when test="$endianness = '1'">option.HardwareEndianness.Little</xsl:when>
                <xsl:when test="$endianness = '2'">option.HardwareEndianness.Big</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:element name="{concat('param.HardwareAtomicIntegerSize.', $newSuffix)}">
            <xsl:variable name="atomicIntegerSize" select="HardwareImplementation/*[name() = concat($oldPrefix, 'LargestAtomicInteger')]" />

            <xsl:choose>
                <xsl:when test="$atomicIntegerSize = '0'">option.HardwareAtomicIntegerSize.Char</xsl:when>
                <xsl:when test="$atomicIntegerSize = '1'">option.HardwareAtomicIntegerSize.Short</xsl:when>
                <xsl:when test="$atomicIntegerSize = '2'">option.HardwareAtomicIntegerSize.Int</xsl:when>
                <xsl:when test="$atomicIntegerSize = '3'">option.HardwareAtomicIntegerSize.Long</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:element name="{concat('param.HardwareAtomicFloatSize.', $newSuffix)}">
            <xsl:variable name="atomicFloatSize" select="HardwareImplementation/*[name() = concat($oldPrefix, 'LargestAtomicFloat')]" />

            <xsl:choose>
                <xsl:when test="$atomicFloatSize = '0'">option.HardwareAtomicFloatSize.None</xsl:when>
                <xsl:when test="$atomicFloatSize = '1'">option.HardwareAtomicFloatSize.Float</xsl:when>
                <xsl:when test="$atomicFloatSize = '2'">option.HardwareAtomicFloatSize.Double</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:element name="{concat('param.HardwareDivisionRounding.', $newSuffix)}">
            <xsl:variable name="divisionRounding" select="HardwareImplementation/*[name() = concat($oldPrefix, 'IntDivRoundTo')]" />

            <xsl:choose>
                <xsl:when test="$divisionRounding = '0'">option.HardwareDivisionRounding.Undefined</xsl:when>
                <xsl:when test="$divisionRounding = '1'">option.HardwareDivisionRounding.Zero</xsl:when>
                <xsl:when test="$divisionRounding = '2'">option.HardwareDivisionRounding.Floor</xsl:when>
            </xsl:choose>
        </xsl:element>

        <xsl:call-template name="hardwareField">
            <xsl:with-param name="oldName" select="concat($oldPrefix, 'ShiftRightIntArith')" />
            <xsl:with-param name="newName" select="concat('param.HardwareArithmeticRightShift.', $newSuffix)" />
        </xsl:call-template>
    </xsl:template>

</xsl:stylesheet>
