/* Copyright 2007-2014 The MathWorks, Inc. */

#include "xil_data_stream.h"
#include "codeinstr_data_stream.h"
#include "xil_interface_lib.h"
#include "comms_interface.h"
#include "codeinstrservice.h"

#ifdef USING_CS_API    
    #include "coderprofile/codeinstr_service/target/CInterface.h"
    #define CODE_INSTR_TX_BUFFER_MEMUNIT_SIZE commsGetMaxPayloadSize()
    #define BUFFER_HEADER_SIZE 0    
#else
    #include "rx_tx_buffer_sizes.h"    
#endif

#define USED_WRITE_BUFFER_SIZE (COMMAND_COMPLETE_SIZE + codeInstrWriteDataAvail)
#define WRITE_BUFFER_SIZE (CODE_INSTR_TX_BUFFER_MEMUNIT_SIZE - COMMAND_COMPLETE_SIZE - BUFFER_HEADER_SIZE)

static void* pBuffer;
static IOUnit_T* codeInstrWriteBuffer;
static IOUnit_T* codeInstrWriteDataPtr;
static uint16_T codeInstrWriteDataAvail;
static MemUnit_T* commandComplete;

/* reset the write buffer */
static XIL_INTERFACE_LIB_ERROR_CODE resetCodeInstrWriteBuffer(void) {
    
    /* Ask the CS to allocate a buffer that the CodeInstr app service will 
     * use for transmission
     */
    if (!commsAllocCodeInstrBuffer(&pBuffer, CODE_INSTR_TX_BUFFER_MEMUNIT_SIZE)) {
        return XIL_INTERFACE_LIB_ERROR;
    }
    codeInstrWriteBuffer = commsGetCodeInstrBufferDataPtr(pBuffer);
    
    /* set commandComplete and codeInstrWriteDataPtr pointer */
    commandComplete = (MemUnit_T *) &codeInstrWriteBuffer[COMMAND_COMPLETE_IDX];
    codeInstrWriteDataPtr = &codeInstrWriteBuffer[WRITE_DATA_BUFFER_IDX];
    
    /* ready for next command */
    codeInstrWriteDataAvail = 0;
    *commandComplete = 0;
    
    return XIL_INTERFACE_LIB_SUCCESS;
}

XIL_INTERFACE_LIB_ERROR_CODE codeInstrInit(void) {    
   
#ifdef USING_CS_API
    /* Create CodeInstr service */
    if (codeInstrServiceTgtCreate(getCommsInstance())!=CODEINSTR_SERVICE_SUCCESS){
        return XIL_INTERFACE_LIB_ERROR;
    }
#endif
    
   /* Reset write buffer */
   if (resetCodeInstrWriteBuffer()!= XIL_INTERFACE_LIB_SUCCESS) {
	   return XIL_INTERFACE_LIB_ERROR;
   }
    
    return XIL_INTERFACE_LIB_SUCCESS;
}

void codeInstrTerminate(void) {      
   
#ifdef USING_CS_API
   /* Destroy CodeInstr service */
   codeInstrServiceTgtDestroy();      
#endif   
}

/* send pending writes */
static XIL_DATA_STREAM_ERROR_CODE sendWriteBuffer(void) {
   /* send */
   if (commsEnqueueBuffer(pBuffer, CODEINSTR_SERVICE_ID, USED_WRITE_BUFFER_SIZE) == COMMS_ERROR) {             
      return XIL_DATA_FLUSH_ERROR;
   }   
   
   /* reset */
   if (resetCodeInstrWriteBuffer()!= XIL_INTERFACE_LIB_SUCCESS) {
	   return XIL_DATA_FLUSH_ERROR;
   }
   
   return XIL_DATA_STREAM_SUCCESS;
}


XIL_DATA_STREAM_ERROR_CODE codeInstrWriteData(const MemUnit_T * src, uint32_T size) {
   XIL_DATA_STREAM_ERROR_CODE errorCode = XIL_DATA_STREAM_SUCCESS;
   const IOUnit_T * srcPtr = (const IOUnit_T *) src;   
   size_t transferAmount;
   uint16_T bufferAvail;
  
   /* block until all data is processed */
   while (size > 0) {      
      /* send if we have a full message worth of data */   
      if (codeInstrWriteDataAvail == WRITE_BUFFER_SIZE) {
         errorCode = sendWriteBuffer();
         if (errorCode != XIL_DATA_STREAM_SUCCESS) {
            return errorCode;
         }
      }
      bufferAvail = WRITE_BUFFER_SIZE - codeInstrWriteDataAvail;
      transferAmount = (uint16_T) MIN(bufferAvail, size);
      /* copy data into write buffer */
      memcpy((void *) codeInstrWriteDataPtr, srcPtr, transferAmount);
      size -= (uint32_T) transferAmount;
      codeInstrWriteDataAvail += (uint16_T) transferAmount;
      srcPtr += transferAmount;
      codeInstrWriteDataPtr += transferAmount;
   }
   return errorCode;
}


XIL_DATA_STREAM_ERROR_CODE codeInstrDataFlush(void) {   

    /* final part of command */
    *commandComplete = 1;

    /* send the write buffer */
    if (sendWriteBuffer() == XIL_DATA_FLUSH_ERROR){
        return XIL_DATA_FLUSH_ERROR;
    }
    
    return XIL_DATA_STREAM_SUCCESS;

}
