/*!
  \file
  This file contains documentation for the 'hello' example.
  It contains some C++ code, to help Doxygen find its way around.
*/
namespace SciGPU {
  namespace Legion {


    /*!
      \page hello A "Hello World" Program
      
      
      \dontinclude hello.cpp

      In this section, we will describe a simple "Hello World!"-type program
      which makes use of the Legion library.
      The program will queue up a set of requests to say Hello, and then
      create a set of \ref Maniple "Maniples" to actually do so.
      
      
      \section helloclass The Task to say "Hello"
      
      The class which we need is extremely simple.
      It is derived from the CPUtask class, and over-rides the
      Task::Perform method.
      \skip class HelloTask : public SciGPU::Legion::CPUtask {
      \until };
      All the Task::Perform method has to do is call the
      \ref Maniple::LogMessage "LogMessage"
      method of the Maniple (accessed through the Task::maniple
      pointer member) with an appropriate string, and then
      return \c true to indicate success.
      It also sleeps for a short time, to give other
      \ref Maniple "Maniples" a chance to acquire work.


      \section enlistlegion Enlisting the Legion

      We now need to create the Legion, and give it some
      \ref Maniple "Maniples" to do some work.
      \skip // Create the Legion
      \until }
      We create a set of anonymous CPUmaniple objects, which
      boost copies when it launches the threads.
      Of course this means that the user <strong>must</strong>
      ensure that any supplied \ref Maniple "Maniples" can
      be copied.
      The CPUmaniple base class satisfies this, but care may
      be required in derived classes.


      \section queuetasks Queuing up the tasks

      At this point in the program, our \ref Maniple "Maniples"
      are in the process of initialising themselves, or are ready
      for work.
      We now need to give them some work to do.
      In this example, we will do this fairly simply, by creating
      a bunch of \c HelloTask objects, and queuing them up
      for the Legion.
      \skip // Declare the list of tasks
      \until }
      One slightly subtle point in the declaration of the
      \c hellos vector: note that we create a sample \c HelloTask
      beforehand, to ensure that the vector we create is fully
      constructed.
      Once the vector is ready, we loop over it, handing a pointer
      to each element off to the Legion.

      The Maniples will begin executing the tasks as soon as the first
      one is enqueued.

      \section hellorun Running the program

      To end this sample program, we need to wait for all the
      Maniples to complete the task list
      \skip // Wait for completion
      \until } while( nComplete < hellos.size() );
      This is a very simplistic method, since we have the
      main thread spin in a loop, checking to see if the
      Legion has completed all the tasks we queued.
      It is good enough for this program, but not in general.

      The full code is in the \c hello example directory.
      When running the program (with two \ref Maniple "Maniples"
      and twenty \c HelloTask objects), we get the following output
      \verbatim
$ ./HelloLegion 
Legion Hello
============

Maniple 0x84d1c00: Starting up
Maniple 0x84d1c00: Hello from a thread!
Maniple 0x84d1c00: Hello from a thread!
Maniple 0x84d1980: Starting up
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1980: Hello from a thread!
Maniple 0x84d1c00: Hello from a thread!
Press ENTER to exit

Maniple 0x84d1980: Caught terminate signal
Maniple 0x84d1980: Shutting down
Maniple 0x84d1c00: Caught terminate signal
Maniple 0x84d1c00: Shutting down
\endverbatim
The Maniple IDs from the log messages will obviously vary from run
to run.
Notice also that the first Maniple was able to start processing
the \c HelloTask list before the second had completed its
initialisation.
    */


  }
}
